
% PolyDPG v1.0 -----------------------------------------------------------

% Discontinuous Petrov-Galerkin (DPG) Anlysis Using Polygonal Elements

% By: Ali Vaziri Astaneh  |    Last Update: 06-16-2017
% URL: www.PolyDPG.com    |    Email: inquiry@PolyDPG.com

% Formulation: A Vaziri Astaneh, F Fuentes, J Mora, L Demkowicz,
%              High-Order Polygonal Finite Elements Using Ultraweak
%              Formulations, ICES REPORT 17-10.

% ------------------------------------------------------------------------

% 2D Poissons Equation on Rectangular Domain with Manufactured Solution

%    - div ( grad(u) )= r       <->      div(q)= r,  q + grad(u)=0

%  Quadrilateral Mesh with Adaptive and Uniform Refinements
%  Triangular    Mesh with              Uniform Refinements
%  Polygonal     Mesh with Adaptive and Uniform Refinements

% INPUT -------------------------------------------------------------------

clear; close all; clc;

% Mesh --------------------------
Pr.MType   = 4;    % Mesh type:  3:Triangular 4:Quadrilateral 5:Polygonal
Pr.nelx    = 1;    % Mumber of elements in x direction
Pr.nely    = 1;    % number of elements in y direction
%                  (accepts only 2,4,8,16 for polygonal mesh)
Pr.RefT    = 2;    % Refinement Type: 1: uniform  2:adaptive
%                  % Uniform ref. only assembles one elem. and saves time
Pr.nRef    = 5;    % Number of refinements
Pr.AdpFac  = 0.25; % Refinement factor for adaptivity
%                   (if residual of an elem > Pr.AdpFac * max(residul) then refine)
% DPG --------------------------
Pr.P       = 2;    % Polynomial order for trial and trace variables
Pr.dP      = 1;    % Polynomial order for test variables
%                    (choose Pr.P + Pr.dP < 21)
Pr.alph=1;         % Adjoint graph norm parameter
% Geometry --------------------------
Pr.Stype   = 2;    % Manufactured Solution 1: smooth 2: local peak
%                  (any solution can be added in "Exact.m")
Pr.Length  = 1;    % Domain length
Pr.Height  = 1;    % Domain height
Pr.DBCtype =...    % Boundary conditions: 1: Dirichlet 2: Neumann
[1 1 1 1];         % [bottom right top left]
% Plot --------------------------
Pr.Pmesh  = 1;     % Plot mesh                       [1 or 0]
Pr.Ptrial = 1;     % Plot trial variables solution   [1 or 0] - time consuming!
Pr.Ptrace = 1;     % Plot trace variables solution   [1 or 0]
Pr.plotE  = 1;     % Plot Error convergence curve    [1 or 0]
Pr.plotR  = 1;     % Plot Residual convergence curve [1 or 0]
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

Err=zeros(1,Pr.nRef+1);  dof=Err; Res=Err;
if (Pr.RefT==1 || Pr.nRef==0)
    for j=1:Pr.nRef+1
        Pr.indx=j;
        if j>1; Pr.nelx=2*Pr.nelx; Pr.nely=2*Pr.nely; end
        [Err(1,j),dof(1,j),Res(1,j)]= DPG2D(Pr);
    end
elseif (Pr.RefT==2 && Pr.nRef>0)
    [Err(1,:),dof(1,:),Res(1,:)]= DPG2D(Pr);
end
if Err(1)~=-1
    Id={'-sk', '-sm', '-sb','-sr', '--ok', ':>b' ,':sb'};
    if (Pr.plotE==1 && Pr.nRef>0)
        figure(15);
        p = polyfit(log10(dof(1,:)),log10(Err(1,:)),1);
        rate=real(2*p(1)); % 2d
        loglog(dof(1,:),Err(1,:),Id{1,1}); hold on
        legend(  ['p=',num2str(Pr.P),', r= ',num2str(ceil(100*rate)/100)], 'Location','southwest')
        set(gca,'color','none');   set(gcf, 'Units', 'centimeters');
        set(gcf, 'Position', [5 5 8.5 7.5]); set(gcf, 'PaperPositionMode', 'auto');
        xlabel('dof'); ylabel('Relative Error')
    end
    if (Pr.plotR==1 && Pr.nRef>0)
        figure(16);
        p = polyfit(log10(dof(1,:)),log10(Res(1,:)),1);
        rate=real(2*p(1)); % 2d
        loglog(dof(1,:),Res(1,:),Id{1,2}); hold on
        legend(  ['p=',num2str(Pr.P),', r= ',num2str(ceil(100*rate)/100)], 'Location','southwest')
        set(gca,'color','none');   set(gcf, 'Units', 'centimeters');
        set(gcf, 'Position', [5 5 8.5 7.5]); set(gcf, 'PaperPositionMode', 'auto');
        xlabel('dof'); ylabel('Relative Residual')
    end
    if Pr.nRef==0
        format SHORTE; disp('');
        disp(' Rel. Err  |   Rel. Residual |  dof  '); disp([Err; Res]);  disp(dof);
    end
end